<?php

namespace TrunkBBI\EagleEye;

use Illuminate\Support\Arr;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class EagleEyeService
{
    private bool $isProd;

    public function __construct()
    {
        $this->isProd = App::environment('production');
    }

    public function getWalletAccount(string $walletId, string $accountId)
    {
        $baseURL = config('eagleeye.endpoints.wallet');

        $uri = '/wallet/'.$walletId.'/account/'.$accountId;

        $response = Http::withHeaders($this->getRequestHeaders($uri))
                ->get($baseURL.$uri);

        if($response->ok()) {
            return $response->object();
        }
        else {
            Log::error('Failed GET to ['.$baseURL.$uri.'] with error ('. $response->status().'): '.$response->body());
        }
    }

    public function getWallet(string $walletId, string | null $status=null, string | null $type= null)
    {
        $baseURL = config('eagleeye.endpoints.wallet');

        $queryParams = [];
        if(isset($status)) {
            $queryParams['status'] = $status;
        }
        if(isset($type)) {
            $queryParams['clientType'] = $type;
        }
        $query = Arr::query($queryParams);

        $uri = '/wallet/'.$walletId.'/accounts'. (!empty($query) ? '?'.$query : '');

        $response = Http::withHeaders($this->getRequestHeaders($uri))
                ->get($baseURL.$uri);

        if($response->ok()) {
            return $response->object();
        }
        else {
            Log::error('Failed GET to ['.$baseURL.$uri.'] with error ('. $response->status().'): '.$response->body());
        }
    }

    public function postWalletAccount(string $walletId, string $campaignId)
    {
        $baseURL = config('eagleeye.endpoints.wallet');

        $uri = '/wallet/'.$walletId.'/campaign/'.$campaignId.'/account';

        $body = json_encode([
            'status' => 'ACTIVE',
            'state' => 'DEFAULT'
        ]);

        $response = Http::withHeaders($this->getRequestHeaders($uri, $body))
                ->withBody($body, 'application/json')
                ->post($baseURL.$uri);

        if($response->created()){
            return $response->object();
        } 
        else {
            Log::error('Failed POST to ['.$baseURL.$uri.'] with error ('. $response->status().'): '.$response->body());
        }
    }

    public function postRedeemAccount(string $walletId, string $accountId)
    {
        $baseURL = config('eagleeye.endpoints.pos');

        $uri = '/connect/account/redeem';

        $body = json_encode([
            'accountId' => $accountId,
            'walletId' => $walletId
        ]);

        $response = Http::withHeaders($this->getRequestHeaders($uri, $body))
                ->withBody($body, 'application/json')
                ->post($baseURL.$uri);

        if($response->ok()){
            return $response->object();
        } 
        else {
            Log::error('Failed POST to ['.$baseURL.$uri.'] with error ('. $response->status().'): '.$response->body());
        }
    }

    public function checkCampaignValid(string $campaignId)
    {
        $baseURL = config('eagleeye.endpoints.resource');

        $uri = '/campaigns/'.$campaignId;

        $response = Http::withHeaders($this->getRequestHeaders($uri))
            ->get($baseURL.$uri);

        if($response->ok() && $response->object()->status == "ACTIVE") {
            return true;
        } else if($response->notFound()) {
            return false;
        }

        return false;
    }

    public function getCampaign(string $campaignId)
    {
        $baseURL = config('eagleeye.endpoints.resource');

        $uri = '/campaigns/'.$campaignId;

        $response = Http::withHeaders($this->getRequestHeaders($uri))
            ->get($baseURL.$uri);

        if($response->ok()) {
            return $response->object();
        } 

        return null;
    }

    private function getRequestHeaders(string $uri, string $body='') : array
    {
        return [
            'Accept' => 'application/json',
            'Accept-Encoding' => 'gzip,deflate',
            'Connection' => 'keep-alive',
            'X-EES-AUTH-CLIENT-ID' => config('eagleeye.api_user'),
            'X-EES-AUTH-HASH' => hash('sha256', $uri.$body.config('eagleeye.api_password'))
        ];
    }
}