<?php

    namespace Tbbi\CF;

    use WP_Query;
    use WP_Error;

class CaterforceImport
{
    private const AKENEO_BASE_URL = 'https://caterforce.cloud.akeneo.com/';
    private const AKENEO_MEDIA_URL = 'https://caterforce.cloud.akeneo.com/media/download/';
    private const AKENO_BRANDS = 'api/rest/v1/reference-entities/brand_entity/records?limit=100';
    private const AKENEO_PRODUCTS = 'api/rest/v1/products?limit=25&search=';
    private const ADMIN_PAGE = 'options-general.php?page=akeneo-pim-menu.php';
    private const AKENO_BRANDS_SEARCH = '{"cf_brand":[{"operator": "IN","value": [%BRANDS%]}]}';

    private $client_token;
    public $brands_array;

    public function __construct()
    {
        $this->client_token = get_option('cf-token');

        add_action('admin_init', function () {
            $this->brands_array = $this->getBrandsArray();
        });

        add_action('admin_post_schedule_import_brands', [$this, 'scheduleBrandsImport'], 10, 1);
        add_action('admin_post_schedule_import', [$this, 'scheduleImport']);
        add_action('admin_post_schedule_import_now', [$this, 'scheduleImportNow'], 10, 1);
        add_action('admin_post_set_brands', [$this, 'setBrands']);
        add_action('cf_import_brands', [$this, 'getBrands'], 10, 1);
        add_action('cf_import_products', [$this, 'getProducts'], 10, 1);
    }

    /**
     * Return brands that are to be used for the API call
     *
     * @return string
     */
    private function getBrandsArray()
    {
        // TODO CHANGE THE INIT ON THIS
        $brands = get_terms([
            'taxonomy' => 'brands',
            'hide_empty' => false,
        ]);

        $brands_required = '';
        foreach ($brands as $brand) {
            if (get_field('brand_used', 'brands_' . $brand->term_id)) {
                $brands_required .= '"' . $brand->slug . '",';
            }
        }
        $brands_required = rtrim($brands_required, ',');

        return str_replace("%BRANDS%", $brands_required, $this::AKENO_BRANDS_SEARCH);
    }

    /**
     * Sets the list of brands that is required
     *
     * @return void
     */
    public function setBrands()
    {
        error_log('setBrands');
        $brands = get_terms([
            'taxonomy' => 'brands',
            'hide_empty' => false,
        ]);

        // first, make all brands not used
        foreach ($brands as $brand) {
            update_field('brand_used', false, 'brands_' . $brand->term_id);
        }

        // now loop and make them used
        foreach ($_POST as $key => $value) {
            if (substr($key, 0, 9) === "cf-brand-") {
                $id = preg_replace('/[^0-9]/', '', $key);
                error_log($id);
                update_field('brand_used', true, 'brands_' . $id);
            }
        }
        wp_redirect(admin_url($this::ADMIN_PAGE));
    }


    /**
     * Runs import brands task in 10 seconds
     *
     * @return void
     */
    public function scheduleBrandsImport($url = '')
    {
        $timestamp = strtotime(date("Y-m-d H:i:s", time() + 10)); // runs 10 sec from now
        error_log('scheduleBrandsImport');
        if ($url !== '') {
            wp_schedule_single_event($timestamp, 'cf_import_brands', array($url), true);
        } else {
            $url = $this::AKENEO_BASE_URL . $this::AKENO_BRANDS;
            wp_schedule_single_event($timestamp, 'cf_import_brands', array($url), true);
        }

        wp_redirect(admin_url($this::ADMIN_PAGE));
        die();
    }

    /**
     * Gets all the brands and add them as terms where required
     *
     * @param [type] $url
     * @return void
     */
    public function getBrands($url)
    {
        error_log("getBrands");
        $args = array(
            'headers' => array(
                'Authorization' => 'Bearer ' . $this->client_token,
                'Content-Type' => 'application/json',
            )
        );

        // Make the call
        $response = wp_remote_get($url, $args);
        if (!$response) {
            return false;
        }

        // Get the content
        $data = json_decode(wp_remote_retrieve_body($response), true);
        if (!isset($data['_embedded']['items']) && is_array($data['_embedded']['items'])) {
            return false;
        }

        foreach ($data['_embedded']['items'] as $brand) {
            $slug = $brand['code'];
            $value = $brand['values']['label'][0]['data'];

            if (!term_exists($slug, 'brands')) {
                wp_insert_term($value, 'brands', ['slug' => $slug]);
            }
        }

        // Next Page
        if (array_key_exists('next', $data['_links'])) {
            error_log("get next");
            $this->scheduleBrandsImport($data['_links']['next']['href']);
        } else {
            error_log("complete");
            die();
        }
    }


    /**
     * Runs import products task in 10 seconds
     *
     * @return void
     */
    public function scheduleImportNow($url = '')
    {
        $timestamp = strtotime(date("Y-m-d H:i:s", time() + 10)); // runs 1 min from now
        if ($url !== '') {
            wp_schedule_single_event($timestamp, 'cf_import_products', array($url), true);
        } else {
            $url = $this::AKENEO_BASE_URL . $this::AKENEO_PRODUCTS . $this->brands_array;
            wp_schedule_single_event($timestamp, 'cf_import_products', array($url), true);
        }

        wp_redirect(admin_url($this::ADMIN_PAGE));
        die();
    }

    /**
     * Runs import task at a predefined period
     *
     * @return void
     */
    public function scheduleImport()
    {
        $time = $_POST['cf-time'];
        update_option('cf-schedule', $time);
    }

    /**
     * Calls the Products endpoint
     *
     * @param [type] $limit
     * @param [type] $page
     * @return void
     */
    public function getProducts($url)
    {
        error_log("getProducts");
        $args = array(
            'headers' => array(
                'Authorization' => 'Bearer ' . $this->client_token,
                'Content-Type' => 'application/json',
            )
        );

        // Make the call
        $response = wp_remote_get($url, $args);
        if (!$response) {
            return false;
        }

        // Get the content
        $data = json_decode(wp_remote_retrieve_body($response), true);
        if (!isset($data['_embedded']['items']) && is_array($data['_embedded']['items'])) {
            return false;
        }

        // parse
        $this->parseProducts($data['_embedded']['items']);

        // Get next
        if (array_key_exists('next', $data['_links'])) {
            error_log("get next");
            $this->scheduleImportNow($data['_links']['next']['href']);
        } else {
            error_log("complete");
            die();
        }
    }

    /**
     * Checks if it is a new or exisiting product
     *
     * @param [type] $data
     * @return void
     */
    private function parseProducts($data)
    {
        $prods = array();

        $query = new WP_Query(array(
            'numberposts' => -1,
            'fields' => 'ids',
            'post_type' => 'members-qas'
        ));
        if ($query->have_posts()) {
            while ($query->have_posts()) {
                $query->the_post();
                $id = get_the_ID();
                $prods[$id] =  get_field('product_identifier', $id);
            }
        }

        // Loop each product to see if it exists, matches on product_identifier
        foreach ($data as $item) {
            if (in_array($item['identifier'], $prods)) {
                $id = array_search($item['identifier'], $prods);
            } else {
                $id = $this->createProduct($item);
            }
            error_log("ID: " . $id);
            $this->updateProduct($id, $item);
        }
    }

    /**
     * Creates a basic product
     *
     * @param [type] $item
     * @return void
     */
    private function createProduct($item)
    {
        $post = array(
            'post_type' => 'members-qas',
            'post_content' => '',
            'post_excerpt' => '',
            'post_status' => 'publish',
            'post_title' => $item['values']['cf_product_name'][0]['data'],
            'post_name' => '',
            'post_date' => $item['created'],
            'post_modified' => $item['updated'],
            'comment_status' => 'closed',
            'post_author' => 1
        );
        $id = wp_insert_post($post, true);
        if (is_wp_error($id)) {
            $errors = $id->get_error_messages();
            foreach ($errors as $error) {
                error_log($error);
            }
            die();
        } else {
            return $id;
        }
    }

    /**
     * Updates a product
     *
     * @param [type] $id
     * @param [type] $item
     * @return void
     */
    private function updateProduct($id, $item)
    {
        error_log("update: " . $item['values']['cf_product_name'][0]['data']);

        // Identifier
        update_field('product_identifier', $item['identifier'], $id);

        // General
        if (array_key_exists('erudus_traded_unit_gtin', $item['values'])) {
            update_field('product_gtin', $item['values']['erudus_traded_unit_gtin'][0]['data'], $id);
        }

        if (array_key_exists('erudus_id', $item['values'])) {
            update_field('erudus_id', $item['values']['erudus_id'][0]['data'], $id);
        }

        if (array_key_exists('erudus_status', $item['values'])) {
            update_field('erudus_status', $item['values']['erudus_status'][0]['data'], $id);
        }

        if (array_key_exists('cf_country_of_origin', $item['values'])) {
            update_field('country_of_origin', $item['values']['cf_country_of_origin'][0]['data'], $id);
        }

        if (array_key_exists('cf_product_description', $item['values'])) {
            update_field('product_overview', $item['values']['cf_product_description'][0]['data'], $id);
        }

        if (array_key_exists('erudus_suppliers_product_code', $item['values'])) {
            update_field('product_supplier_code', $item['values']['erudus_suppliers_product_code'][0]['data'], $id);
        }

        if (array_key_exists('erudus_supplier_name', $item['values'])) {
            update_field('product_supplier', $item['values']['erudus_supplier_name'][0]['data'], $id);
        }

        if (array_key_exists('erudus_storage_instructions', $item['values'])) {
            update_field('product_storage', $item['values']['erudus_storage_instructions'][0]['data'], $id);
        }

        if (array_key_exists('erudus_directions_for_use', $item['values'])) {
            update_field('product_directions', $item['values']['erudus_directions_for_use'][0]['data'], $id);
        }

        if (array_key_exists('erudus_ingredients', $item['values'])) {
            update_field('product_ingredients', $item['values']['erudus_ingredients'][0]['data'], $id);
        }

        // Brand
        if (array_key_exists('cf_brand', $item['values'])) {
            $brand_slug = $item['values']['cf_brand'][0]['data'];
            wp_set_object_terms($id, $brand_slug, 'brands');
        }

        // Allergens
        $allergens = array(
            'cf_contains_sulphur_sulphites' => 'Sulphites',
            'cf_contains_soybeans' => 'Soybeans',
            'cf_contains_crustacea' => 'Crustaea',
            'cf_contains_molluscs' => 'Molluscs',
            'cf_contains_eggs' => 'Eggs',
            'cf_contains_nuts' => 'Nuts',
            'cf_contains_mustard' => 'Mustard',
            'cf_contains_cereals' => 'Cereals',
            'cf_contains_peanuts' => 'Peanuts',
            'cf_contains_milk' => 'Milk',
            'cf_contains_lupin' => 'Lupin',
            'cf_contains_sesame' => 'Seasame',
            'cf_contains_fish' => 'Fish',
            'cf_celery_celeriac' => 'Celerey'
        );
        $this->updateTaxTerms('allergens', $allergens, $item, $id);

        // Diet
        $diet = array(
            'cf_halal_diet' => 'Halal',
            'cf_kosher_diet' => 'Kosher',
            'cf_suitable_vegan' => 'Vegan',
            'cf_suitable_vegetarian' => 'Vegetarian',
            'cf_suitable_coeliacs' => 'Coeliac',
            'cf_suitable_lactose' => 'Lactose'
        );
        $this->updateTaxTerms('diet', $diet, $item, $id);

        // Category
        if (array_key_exists('erudus_product_category', $item['values'])) {
            $category = $item['values']['erudus_product_category'][0]['data'];
            $cat_slug = trim(strtolower($category));
            if (!term_exists($cat_slug, 'product-category')) {
                wp_insert_term($category, 'product-category', ['slug' => $cat_slug]);
            }
            wp_set_object_terms($id, $cat_slug, 'product-category');
        }

        // Images
        $this->updateImages($item, $id);
    }

    /**
     * Updates taxonomy terms associated with a product
     *
     * @param [type] $tax
     * @param [type] $array
     * @param [type] $item
     * @param [type] $id
     * @return void
     */
    private function updateTaxTerms($tax, $array, $item, $id)
    {
        foreach ($array as $key => $value) {
            if (array_key_exists($key, $item['values'])) {
                if ($item['values'][$key][0]['data'] == 'yes') {
                    $slug = trim(strtolower($value));
                    if (!term_exists($slug, $tax)) {
                        wp_insert_term($value, $tax, ['slug' => $slug]);
                    }
                    wp_set_object_terms($id, $slug, $tax);
                }
            }
        }
    }

    /**
     * Simple method to see if an image exists in the db
     *
     * @param [type] $filename
     * @return void
     */
    private function checkForFile($filename)
    {
        global $wpdb;
        return intval($wpdb->get_var("SELECT post_id FROM {$wpdb->postmeta} WHERE meta_value LIKE '%/$filename'"));
    }

    /**
     * Uploads images to the media library
     *
     * @param [type] $item
     * @param [type] $id
     * @return void
     */
    public function updateImages($item, $id)
    {
        // TODO, check these...
        require_once('wp-load.php');
        require_once('wp-admin/includes/file.php');
        require_once('wp-admin/includes/media.php');
        require_once('wp-admin/includes/image.php');

        $admin_path = str_replace(get_bloginfo('url') . '/', ABSPATH, get_admin_url());
        error_log("update images");

        $keys = array('web_image_1', 'web_image_2', 'web_image_3', 'web_image_4');
        $attachement_ids = array();
        foreach ($keys as $key) {
            if (array_key_exists($key, $item['values'])) {
                // Because the Akeneo PIM system does a double encode on the slashes in the filename...
                $url = str_replace("/", "%252F", $item['values'][$key][0]['data']);
                $url = $this::AKENEO_MEDIA_URL . $url;

                if ($this->checkForFile(basename($url)) == null) {
                    $tmp_file = array();
                    $tmp_file['name'] = basename($url);
                    $tmp_file['tmp_name'] = download_url($url);
                    $attachment_id = media_handle_sideload($tmp_file);
// Todo - error checking
                } else {
                    $attachment_id = $this->checkForFile(basename($url));
                }

                // Add to arr
                array_push($attachement_ids, $attachment_id);
            }
        }

        // Delete all the current rows
        $images = get_field('product_images', $id);
        if (is_array($images)) {
            $i = 1;
            foreach ($images as $image) {
                delete_row('images', $i);
                $i++;
            }
        }

        // Add
        foreach ($attachement_ids as $attach) {
            $row = array('image' => $attach);
            add_row('product_images', $row, $id);
        }
    }

    /**
     * This will delete the product at somepoint
     *
     * @return void
     */
    private function deleteProduct()
    {
    }
}
