<?php

namespace Tbbi\CF;

use WP_Query;
use WP_Error;

class CaterforceAuth
{
    private const AKENEO_BASE_URL = 'https://caterforce-group.cloud.akeneo.com/';
    private const AKENEO_AUTH = 'api/oauth/v1/token';
    private const AKENO_CATEGORIES = 'api/rest/v1/categories?limit=100';
    private const AKENO_BRANDS = 'api/rest/v1/reference-entities/brand_entity/records?limit=100';
    private const AKENEO_PRODUCTS = 'api/rest/v1/products?limit=10&search=';
    private const AKENEO_PRODUCT = 'api/rest/v1/products/';
    private const ADMIN_PAGE = 'edit.php?post_type=qas&page=akeneo-pim-menu';
    private const AKENO_BRANDS_SEARCH = '{"cf_brand":[{"operator": "IN","value": [%BRANDS%]}]}';

    private $client_id;
    private $client_secret;
    private $client_refresh;
    private $client_token;
    private $email_list;
    private $debug;

    public $brands_array;

    public $category_counter = 0;
    public $brand_counter = 0;
    public $product_counter = 0;

    public function __construct()
    {
        $this->client_id = get_option('cf-id');
        $this->client_secret = get_option('cf-secret');
        $this->client_refresh = get_option('cf-refresh');
        $this->client_token = get_option('cf-token');
        $this->email_list = get_option('cf-emails');

        $this->debug = get_option('cf-debug');

        add_filter('cron_schedules', [$this, 'setCronSchedules']);
        if (!wp_next_scheduled('cf_renew_token')) {
            wp_schedule_event(time(), '5min', 'cf_renew_token');
        }

        add_action('admin_init', function () {
            $this->brands_array = $this->getBrandsArray();
        });

        add_action('admin_post_get_cf_token', [$this, 'getOauthToken']);
        add_action('admin_post_renew_cf_token', [$this, 'renewOauthToken']);
        add_action('admin_post_destroy_cf_token', [$this, 'destroyOauthToken']);
        add_action('admin_post_set_debug', [$this, 'setDebug']);
        add_action('admin_post_set_cf_emails', [$this, 'setEmails']);

        add_action('admin_post_schedule_import_brands', [$this, 'scheduleBrandsImport'], 10, 2);
        add_action('admin_post_schedule_import', [$this, 'scheduleProductsImport'], 10, 2);
        add_action('admin_post_schedule_import_now', [$this, 'scheduleProductsImport'], 10, 2);
        add_action('admin_post_set_brands', [$this, 'setBrands']);
        add_action('admin_post_schedule_import_categories', [$this, 'scheduleCategoriesImport']);
        add_action('admin_post_schedule_new_product', [$this, 'getNewProduct']);

        add_action('cf_import_brands', [$this, 'getBrands'], 10, 2);
        add_action('cf_import_products', [$this, 'getProducts'], 10, 2);
        add_action('cf_schedule_import_products', [$this, 'getProducts'], 10, 2);
        add_action('cf_import_categories', [$this, 'getCategories'], 10, 2);
        add_action('cf_renew_token', [$this, 'renewOauthToken'], 10, 1);

        add_action('wp_ajax_cf_import_single', [$this, 'getSingleProduct']);
    }

    ////////////////////////////////////
    //// HELPERS
    ////////////////////////////////////
    private function sendEmail($subject, $message)
    {
        $headers = "From: website@caterforce.co.uk\r\n" . "Reply-To: rich.jones-smith@trunkbbi.com\r\n";
        $timestamp = date("Y-m-d H:i:s");
        $sent = wp_mail($this->email_list, $subject, $message . ' - Finished: '. $timestamp, $headers);
        $this->debug("Email: ". $sent);
    }

    public function setEmails()
    {
        $redirect = false;
        if (!isset($_POST['cf-emails'])) {
            wp_redirect(admin_url($this::ADMIN_PAGE));
        }

        update_option('cf-emails', $_POST['cf-emails']);

        wp_redirect(admin_url($this::ADMIN_PAGE));
    }

    public function setDebug()
    {
        if(isset($_POST['cf-debug'])) {
            update_option('cf-debug', $_POST['cf-debug']);
        } else {
            update_option('cf-debug', '');
        }
        wp_redirect(admin_url($this::ADMIN_PAGE));
    }

    public function debug($message)
    {
        if($this->debug) {
            error_log($message);
        }
    }

    private function downloadFile($url)
    {
        $this->debug("* Download: ".$url);
        $curl = curl_init();
        curl_setopt_array($curl, array(
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_ENCODING => "",
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_TIMEOUT => 0,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_CUSTOMREQUEST => "GET",
            CURLOPT_HTTPHEADER => array(
                "Authorization: Bearer " . $this->client_token
            ),
        ));

        $response = curl_exec($curl);
        $httpcode = curl_getinfo($curl, CURLINFO_HTTP_CODE);

        curl_close($curl);

        return $response;
    }

    ////////////////////////////////////
    //// AUTH
    ////////////////////////////////////
    public function setCronSchedules($schedules){
        if(!isset($schedules["5min"])){
            $schedules["5min"] = array(
                'interval' => 5*60,
                'display' => __('Once every 5 minutes'));
        }
        return $schedules;
    }

    private function makeRequest($body, $endpoint)
    {
        $this->debug("makeRequest: " . $endpoint);
        $headers = array(
            'Authorization' => 'Basic ' . base64_encode($this->client_id . ':' . $this->client_secret),
            'Content-Type' => 'application/json',
        );

        $args = array(
            'headers' => $headers,
            'body' => $body
        );

        return wp_remote_post($this::AKENEO_BASE_URL . $endpoint, $args);
    }

    /**
     * Creates the initial oauth token
     */
    public function getOauthToken()
    {
        $redirect = false;
        if (isset($_POST['redirect']) && $_POST['redirect'] == 'true') {
            $redirect = true;
        }

        if (!isset($_POST['cf-username']) || !isset($_POST['cf-password'])) {
            if ($redirect) {
                wp_redirect(admin_url($this::ADMIN_PAGE));
            }
            die();
        }

        $username = $_POST['cf-username'];
        $password = $_POST['cf-password'];

        update_option('cf-id', $_POST['cf-id']);
        update_option('cf-secret', $_POST['cf-secret']);

        $this->client_id = get_option('cf-id');
        $this->client_secret = get_option('cf-secret');

        $body = json_encode(array(
            'username' => $username,
            'password' => $password,
            'grant_type' => 'password',
        ));
        $response = $this->makeRequest($body, $this::AKENEO_AUTH);

        if (!$response) {
            if ($redirect) {
                wp_redirect(admin_url($this::ADMIN_PAGE));
            }
            die();
        }

        $data = json_decode(wp_remote_retrieve_body($response), true);
        if (!isset($data['access_token'], $data['expires_in'], $data['refresh_token'])) {
            if ($redirect) {
                wp_redirect(admin_url($this::ADMIN_PAGE));
            }
            die();
        }

        update_option('cf-token', $data['access_token']);
        update_option('cf-refresh', $data['refresh_token']);

        // Front end
        if ($redirect) {
            wp_redirect(admin_url($this::ADMIN_PAGE));
            die();
        }
    }

    /**
     * Renews the oauth token
     */
    public function renewOauthToken()
    {
        $this->debug('renewOauthtoken');
        $redirect = false;
        if (isset($_POST['redirect']) && $_POST['redirect'] == 'true') {
            $redirect = true;
        }

        $body = json_encode(array(
            'refresh_token' => $this->client_refresh,
            'grant_type' => 'refresh_token',
        ));

        $response = $this->makeRequest($body, $this::AKENEO_AUTH);

        if (!$response) {
            if ($redirect) {
                wp_redirect($this::ADMIN_PAGE);
            }
            die();
        }

        $data = json_decode(wp_remote_retrieve_body($response), true);

        if (!isset($data['access_token'], $data['expires_in'], $data['refresh_token'])) {
            if ($redirect) {
                wp_redirect(admin_url($this::ADMIN_PAGE));
            }
            die();
        }

        update_option('cf-token', $data['access_token']);
        update_option('cf-refresh', $data['refresh_token']);

        $this->client_refresh = get_option('cf-refresh');
        $this->client_token = get_option('cf-token');

        // Front end
        if ($redirect) {
            wp_redirect(admin_url($this::ADMIN_PAGE));
            die();
        }
    }

    /**
     * Deletes all oauth tokens and secrets
     */
    public function destroyOauthToken()
    {
        $redirect = false;
        if (isset($_POST['redirect']) && $_POST['redirect'] == 'true') {
            $redirect = true;
        }

        update_option('cf-id', null);
        update_option('cf-secret', null);
        update_option('cf-token', null);
        update_option('cf-refresh', null);

        // Front end
        if ($redirect) {
            wp_redirect(admin_url('options-general.php?page=akeneo-pim-menu.php'));
            die();
        }
    }

    ////////////////////////////////////
    //// CATEGORIES
    ////////////////////////////////////
    public function scheduleCategoriesImport($url = '', $count = 0)
    {
        $timestamp = strtotime(date("Y-m-d H:i:s", time() + 10)); // runs 10 sec from now
        if ($url !== '') {
            wp_schedule_single_event($timestamp, 'cf_import_categories', array($url, $count), true);
        } else {
            $url = $this::AKENEO_BASE_URL . $this::AKENO_CATEGORIES;
            wp_schedule_single_event($timestamp, 'cf_import_categories', array($url, $count), true);
            wp_redirect(admin_url($this::ADMIN_PAGE));
        }
        die();
    }

    public function getCategories($url, $count)
    {
        $this->category_counter = $count;

        $this->debug("getCategories");
        $this->debug($url);
        $args = array(
            'headers' => array(
                'Authorization' => 'Bearer ' . $this->client_token,
                'Content-Type' => 'application/json',
            )
        );

        $response = wp_remote_get($url, $args);
        if (!$response) {
            $this->debug("FAIL");
            return false;
        }

        if (wp_remote_retrieve_response_code($response) != 200) {
            $this->renewOauthToken();
            $this->scheduleCategoriesImport($url);
            die();
        }

        // Get the content
        $data = json_decode(wp_remote_retrieve_body($response), true);

        $this->parseCategories($data['_embedded']['items']);

        // Get next
        if (array_key_exists('next', $data['_links'])) {
            $this->debug("get next");
            $this->scheduleCategoriesImport($data['_links']['next']['href'], $this->category_counter);
        } else {
            $this->sendEmail('Caterforce Import Complete', 'Category Import Complete. Categories Parsed:' . $this->category_counter);
            $this->debug("complete");
            $this->category_counter = 0;
            die();
        }
    }

    public function parseCategories($data)
    {
        $this->debug("parseCategories");

        $parent_slug = 'own_brand';
        $parents_allowed = array(
            'own_brand',
            'ob_grocery',
            'ob_dessert',
            'ob_meat_poultry',
            'ob_accompaniments',
            'ob_bakery',
            'ob_chilled_dairy',
            'ob_fish_seafood',
            'ob_non_food',
            'ob_beverages',
            'ob_non_food'
        );

        foreach ($data as $item) {
            $cat_name = $item['labels']['en_GB'];
            $cat_slug = trim(strtolower($item['code']));
            $cat_parent = trim(strtolower($item['parent']));
            $this->debug('*' . $cat_name);

            if ($cat_slug == $parent_slug) {
                if (!term_exists($cat_name, 'product-category')) {
                    wp_insert_term($cat_name, 'product-category', ['slug' => $cat_slug]);
                }
            }

            if (in_array($cat_parent,$parents_allowed)) {
                $parent = get_term_by('slug', $cat_parent, 'product-category');
                if (!term_exists($cat_slug, 'product-category')) {
                    wp_insert_term($cat_name, 'product-category', ['slug' => $cat_slug, 'parent' => $parent->term_id]);
                }
            }
            
            $this->category_counter++;
        }
    }

    ////////////////////////////////////
    ////  BRANDS
    ////////////////////////////////////
    public function scheduleBrandsImport($url = '', $count = 0)
    {
        $this->debug("scheduleBrandsImport");
        $timestamp = strtotime(date("Y-m-d H:i:s", time() + 10)); // runs 10 sec from now
        if ($url !== '') {
            wp_schedule_single_event($timestamp, 'cf_import_brands', array($url, $count), true);
        } else {
            $url = $this::AKENEO_BASE_URL . $this::AKENO_BRANDS;
            wp_schedule_single_event($timestamp, 'cf_import_brands', array($url, $count), true);
        }
        wp_redirect(admin_url($this::ADMIN_PAGE));
        die();
    }

    /**
     * Gets all the brands from Akeno PIM
     */
    public function getBrands($url, $count)
    {
        $this->debug("getBrands");
        $this->debug($url);
        $this->brands_counter = $count;

        $args = array(
            'headers' => array(
                'Authorization' => 'Bearer ' . $this->client_token,
                'Content-Type' => 'application/json',
            )
        );

        $response = wp_remote_get($url, $args);
        if (!$response) {
            return false;
        }

        if (wp_remote_retrieve_response_code($response) != 200) {
            $this->renewOauthToken();
            $this->scheduleBrandsImport($url);
            die();
        }

        $data = json_decode(wp_remote_retrieve_body($response), true);
        if (!isset($data['_embedded']['items']) && is_array($data['_embedded']['items'])) {
            return false;
        }

        foreach ($data['_embedded']['items'] as $brand) {
            $slug = $brand['code'];
            $value = $brand['values']['label'][0]['data'];
            $this->debug("* ".$value);

            if (!term_exists($slug, 'brands')) {
                wp_insert_term($value, 'brands', ['slug' => $slug]);
            }

            $this->brands_counter++;
        }

        if (array_key_exists('next', $data['_links'])) {
            $this->scheduleBrandsImport($data['_links']['next']['href'], $this->brands_counter);
        } else {
            $this->sendEmail('Caterforce Import Complete', 'Brand Import Complete. Brands Parsed:' . $this->brands_counter);
            $this->debug("complete brands");
            $this->brands_counter = 0;
            die();
        }
    }

    /**
     * Gets list of brands for use
     *
     * @return void
     */
    public function getBrandsArray()
    {
        $brands_required = '';
        $brands = get_terms([
            'taxonomy' => 'brands',
            'hide_empty' => false,
        ]);
        foreach ($brands as $brand) {
            if (get_field('brand_used', 'brands_' . $brand->term_id)) {
                $brands_required .= '"' . $brand->slug . '",';
            }
        }
        $brands_required = rtrim($brands_required, ',');

        return str_replace("%BRANDS%", $brands_required, $this::AKENO_BRANDS_SEARCH);
    }

    /**
     * Sets the brands in use
     */
    public function setBrands()
    {
        $brands = get_terms([
            'taxonomy' => 'brands',
            'hide_empty' => false,
        ]);

        // set All to false
        foreach ($brands as $brand) {
            update_field('brand_used', false, 'brands_' . $brand->term_id);
        }

        // Now set the ones we want
        $use_array = array();
        foreach ($_POST as $key => $value) {
            if (substr($key, 0, 9) === "cf-brand-") {
                $id = preg_replace('/[^0-9]/', '', $key);
                update_field('brand_used', true, 'brands_' . $id);
                array_push($use_array, $id);
            }
        }

        // Now we need to delete all products that are not set as a brand to be used
        $delete_array = array();
        $delete_query = new WP_Query(array(
            'post_type' => 'qas',
            'posts_per_page' => -1,
            'tax_query' => array(
                'relation' => 'AND',
                array(
                    'taxonomy' => 'brands',
                    'field' => 'term_id',
                    'terms' =>  $use_array,
                    'operator' => 'NOT IN'
                )
            )
        ));

        if ($delete_query->have_posts()) {
            while($delete_query->have_posts()) {
                $delete_query->the_post();
                array_push($delete_array, get_the_ID());
            }
        }

        if(count($delete_array) > 0) {
            foreach($delete_array as $id) {
                $this->deleteProduct($id);
            }
        }

        wp_redirect(admin_url($this::ADMIN_PAGE));
    }

    ////////////////////////////////////
    // PRODUCTS
    ////////////////////////////////////
    public function scheduleProductsImport($url = '', $count = 0)
    {
        $this->debug("scheduleProductsImport");
        if (isset($_POST['cf-time']) && $_POST['cf-time']) {
            $time = $_POST['cf-time'];
            update_option('cf-schedule', $time);
            wp_clear_scheduled_hook('cf_schedule_import_products');

            if (!wp_next_scheduled('cf_schedule_import_products', array($url))) {
                $url = $this::AKENEO_BASE_URL . $this::AKENEO_PRODUCTS . $this->brands_array;
                wp_schedule_event(strtotime($time), 'daily', 'cf_schedule_import_products', array($url, $count), true);
                wp_redirect(admin_url($this::ADMIN_PAGE));
            }
            $this->sendEmail('Caterforce Import Complete', 'Product Import Scheduled for' . $time . ' daily.');
        } else {
            $timestamp = strtotime(date("Y-m-d H:i:s", time() + 10));
            if ($url !== '') {
                wp_schedule_single_event($timestamp, 'cf_import_products', array($url, $count), true);
            } else {
                $url = $this::AKENEO_BASE_URL . $this::AKENEO_PRODUCTS . $this->brands_array;
                wp_schedule_single_event($timestamp, 'cf_import_products', array($url, $count), true);
                wp_redirect(admin_url($this::ADMIN_PAGE));
            }
        }
    }

    public function getNewProduct()
    {
        $this->debug("getNewProduct");

        $product_id = $_POST['cf-new'];
        $timestamp = strtotime(date("Y-m-d H:i:s", time() + 10));
        $url = $this::AKENEO_BASE_URL . $this::AKENEO_PRODUCT . $product_id;
        wp_schedule_single_event($timestamp, 'cf_import_products', array($url, 0), true);
        wp_redirect(admin_url($this::ADMIN_PAGE));
    }

    public function getSingleProduct()
    {
        $this->debug("getSingleProduct");

        if (!isset($_POST['hash']) || !isset($_POST['post_id']) || !isset($_POST['product_id'])) {
            die();
        } else {
            $post_id = $_POST['post_id'];
            $product_id = $_POST['product_id'];
            $hash = $_POST['hash'];

            if (!wp_verify_nonce($hash, $post_id.'nfjklgsdnjklvnjklvsbnilv')) {
                die();
            }

            $timestamp = strtotime(date("Y-m-d H:i:s", time() + 10));
            $url = $this::AKENEO_BASE_URL . $this::AKENEO_PRODUCT . $product_id;
            wp_schedule_single_event($timestamp, 'cf_import_products', array($url, 0), true);
            die();
        }
    }

    public function getProducts($url, $count)
    {
        $this->debug("getProducts");
        $this->debug($url);
        $this->products_counter = $count;

        $args = array(
            'headers' => array(
                'Authorization' => 'Bearer ' . $this->client_token,
                'Content-Type' => 'application/json',
            )
        );

        $response = wp_remote_get($url, $args);
        if (!$response) {
            return false;
        }

        if (wp_remote_retrieve_response_code($response) != 200) {
            $this->renewOauthToken();
            $this->scheduleProductsImport($url);
            die();
        }

        // Get the content
        $data = json_decode(wp_remote_retrieve_body($response), true);

        // Is it single?
        if (!isset($data['_embedded']['items'])) {
            $this->debug("SINGLE");
            $this->parseProducts(array($data));
            $this->sendEmail('Caterforce Import Complete', 'Single Product Import Complete');
            $this->products_counter = 0;
            die();
        } else {
            $this->parseProducts($data['_embedded']['items']);
    
            // Get next
            if (array_key_exists('next', $data['_links'])) {
                $this->debug("get next");
                $this->scheduleProductsImport($data['_links']['next']['href'], $this->products_counter);
            } else {
                $this->sendEmail('Caterforce Import Complete', 'Product Import Complete. Products Parsed:' . $this->products_counter);
                $this->debug("complete");
                $this->products_counter = 0;
                die();
            }
        }
    }

    private function parseProducts($data)
    {
        $this->debug("parseProducts");
        $prods = array();

        $query = new WP_Query(array(
            'posts_per_page' => -1,
            'fields' => 'ids',
            'post_type' => 'qas'
        ));
        if ($query->have_posts()) {
            while ($query->have_posts()) {
                $query->the_post();
                $id = get_the_ID();
                $prods[$id] = get_field('product_identifier', $id);
            }
        }

        // Loop each product to see if it exists, matches on product_identifier
        foreach ($data as $item) {
            // Do we sync

            if (array_key_exists('cf_website_sync', $item['values'])) {
                if ($item['values']['cf_website_sync'][0]['data']) {
                    // Yes
                    if (in_array($item['identifier'], $prods)) {
                        $id = array_search($item['identifier'], $prods);
                    } else {
                        $id = $this->createProduct($item);
                    }
                    $this->debug("ID: " . $id);
                    $this->updateProduct($id, $item);
                } else {
                    // No
                    if (in_array($item['identifier'], $prods)) {
                        
                        $id = array_search($item['identifier'], $prods);
                        $this->debug("DELETEME: ".$id);
                        $this->deleteProduct($id);
                    }
                }
            } else {
                // Catch if the key doesnt exist
                /* if (in_array($item['identifier'], $prods)) {
                    $id = array_search($item['identifier'], $prods);
                } else {
                    $id = $this->createProduct($item);
                }
                $this->debug("ID: " . $id);
                $this->updateProduct($id, $item); */

                // do not mutate product data unless cf_website_sync is set
            }

            $this->products_counter++;
        }
    }

    private function deleteProduct($id)
    {
        wp_delete_post($id);
    }

    private function createProduct($item)
    {
        $this->debug("createProduct");
        $post = array(
            'post_type' => 'qas',
            'post_content' => '',
            'post_excerpt' => '',
            'post_status' => 'publish',
            'post_title' => $item['values']['cf_product_name'][0]['data'],
            'post_name' => '',
            'post_date' => $item['created'],
            'post_modified' => $item['updated'],
            'comment_status' => 'closed',
            'post_author' => 1
        );
        $id = wp_insert_post($post, true);
        if (is_wp_error($id)) {
            $errors = $id->get_error_messages();
            foreach ($errors as $error) {
                $this->debug($error);
            }
            die();
        } else {
            return $id;
        }
    }

    private function updateProduct($id, $item)
    {
        $this->debug("update: " . $item['values']['cf_product_name'][0]['data']);

        // Title
        if ($item['values']['cf_product_name'][0]['data'] != get_the_title($id)) {
            $update = array(
                'ID' => $id,
                'post_title' => $item['values']['cf_product_name'][0]['data']
            );
            wp_update_post($update);
        }

        // Identifier
        update_field('product_identifier', $item['identifier'], $id);

        // General
        if (array_key_exists('erudus_traded_unit_gtin', $item['values'])) {
            update_field('product_gtin', $item['values']['erudus_traded_unit_gtin'][0]['data'], $id);
        }

        if (array_key_exists('cf_csbc', $item['values'])) {
            update_field('product_csbc', $item['values']['cf_csbc'][0]['data'], $id);
        }

        if (array_key_exists('erudus_id', $item['values'])) {
            update_field('erudus_id', $item['values']['erudus_id'][0]['data'], $id);
        }

        if (array_key_exists('erudus_status', $item['values'])) {
            update_field('erudus_status', $item['values']['erudus_status'][0]['data'], $id);
        }

        if (array_key_exists('cf_country_of_origin', $item['values'])) {
            update_field('country_of_origin', $item['values']['cf_country_of_origin'][0]['data'], $id);
        }

        if (array_key_exists('cf_product_description', $item['values'])) {
            update_field('product_overview', $item['values']['cf_product_description'][0]['data'], $id);
        }

        if (array_key_exists('cf_product_description2', $item['values'])) {
            update_field('product_overview_2', $item['values']['cf_product_description2'][0]['data'], $id);
        }

        if (array_key_exists('erudus_suppliers_product_code', $item['values'])) {
            update_field('product_supplier_code', $item['values']['erudus_suppliers_product_code'][0]['data'], $id);
        }

        if (array_key_exists('erudus_supplier_name', $item['values'])) {
            update_field('product_supplier', $item['values']['erudus_supplier_name'][0]['data'], $id);
        }

        if (array_key_exists('erudus_storage_instructions', $item['values'])) {
            update_field('product_storage', $item['values']['erudus_storage_instructions'][0]['data'], $id);
        }

        if (array_key_exists('erudus_directions_for_use', $item['values'])) {
            update_field('product_directions', $item['values']['erudus_directions_for_use'][0]['data'], $id);
        }

        if (array_key_exists('erudus_ingredients', $item['values'])) {
            update_field('product_ingredients', $item['values']['erudus_ingredients'][0]['data'], $id);
        }

        if (array_key_exists('cf_case_size', $item['values'])) {
            update_field('case_size', $item['values']['cf_case_size'][0]['data'], $id);
        }  

        // Nutrition
        if (array_key_exists('cf_kj_per_100', $item['values'])) {
            update_field('kj_per_100g', $item['values']['cf_kj_per_100'][0]['data'], $id);
        }

        if (array_key_exists('cf_fat_per_100', $item['values'])) {
            update_field('fat_per_100g', $item['values']['cf_fat_per_100'][0]['data'], $id);
        }

        if (array_key_exists('cf_fat_saturates_per_100', $item['values'])) {
            update_field('fat_saturates_per_100g', $item['values']['cf_fat_saturates_per_100'][0]['data'], $id);
        }

        if (array_key_exists('cf_kcal_per_100', $item['values'])) {
            update_field('kcal_per_100g', $item['values']['cf_kcal_per_100'][0]['data'], $id);
        }

        if (array_key_exists('cf_salt_per_100', $item['values'])) {
            update_field('salt_per_100g', $item['values']['cf_salt_per_100'][0]['data'], $id);
        }

        if (array_key_exists('cf_fibre_per_100', $item['values'])) {
            update_field('fibre_per_100g', $item['values']['cf_fibre_per_100'][0]['data'], $id);
        }

        if (array_key_exists('cf_carbs_per_100g', $item['values'])) {
            update_field('carbs_per_100g', $item['values']['cf_carbs_per_100g'][0]['data'], $id);
        }

        if (array_key_exists('cf_carbs_sugars_per_100g', $item['values'])) {
            update_field('carbs_that_sugars_per_100g', $item['values']['cf_carbs_sugars_per_100g'][0]['data'], $id);
        }

        if (array_key_exists('cf_protein_per_100', $item['values'])) {
            update_field('protein_per_100g', $item['values']['cf_protein_per_100'][0]['data'], $id);
        }

        if (array_key_exists('video_html', $item['values'])) {
            update_field('product_video_html', $item['values']['video_html'][0]['data'], $id);
        }

        $this->updateVideoThumbnailImage($item, $id);

        // Brand
        if (array_key_exists('cf_brand', $item['values'])) {
            $brand_slug = $item['values']['cf_brand'][0]['data'];
            wp_set_object_terms($id, $brand_slug, 'brands');
        }

        // Allergens
        $allergens = array(
            'cf_contains_sulphur_sulphites' => 'Sulphur Dioxide (Sulphites)',
            'cf_contains_soybeans' => 'Soya',
            'cf_contains_crustacea' => 'Crustaceans',
            'cf_contains_molluscs' => 'Molluscs',
            'cf_contains_eggs' => 'Eggs',
            'cf_contains_nuts' => 'Nuts (Tree nuts)',
            'cf_contains_mustard' => 'Mustard',
            'cf_contains_cereals' => 'Cereals Containing Gluten',
            'cf_contains_peanuts' => 'Peanuts',
            'cf_contains_milk' => 'Milk',
            'cf_contains_lupin' => 'Lupin',
            'cf_contains_sesame' => 'Sesame',
            'cf_contains_fish' => 'Fish',
            'cf_celery_celeriac' => 'Celery',
            'cf_no_allergens' => 'No allergens'
        );

        // Contains Allergens
        $current_terms = get_the_terms($id, 'allergens');
        if (is_array($current_terms)) {
            foreach ($current_terms as $term) {
                wp_remove_object_terms($id, $term->term_id, 'allergens');
            }
        }
        if (array_key_exists('contains_allergens', $item['values'])) {
            $terms = array();
            foreach ($item['values']['contains_allergens'][0]['data'] as $term) {
                if (!term_exists($term, 'allergens')) {
                    wp_insert_term($allergens[$term], 'allergens', ['slug' => $term]);
                }
                array_push($terms, $term);
            }
            wp_set_post_terms($id, $terms, 'allergens');
        }

        // May Contain Allergens
        $current_terms = get_the_terms($id, 'allergens-may-contain');
        if (is_array($current_terms)) {
            foreach ($current_terms as $term) {
                wp_remove_object_terms($id, $term->term_id, 'allergens-may-contain');
            }
        }
        if (array_key_exists('may_contain_allergens', $item['values'])) {
            $terms = array();
            foreach ($item['values']['may_contain_allergens'][0]['data'] as $term) {
                if (!term_exists($term, 'allergens-may-contain')) {
                    wp_insert_term($allergens[$term], 'allergens-may-contain', ['slug' => $term]);
                }
                array_push($terms, $term);
            }
            wp_set_post_terms($id, $terms, 'allergens-may-contain');
        }

        // Diet
        $diet = array(
            'cf_halal_diet' => 'Halal Approved',
            'cf_kosher_diet' => 'Kosher Approved',
            'cf_suitable_vegan' => 'Suitable for Vegan Diets',
            'cf_suitable_vegetarian' => 'Suitable for Vegetarian Diets',
            'cf_suitable_coeliacs' => 'Suitable for Gluten-Free Diets',
            'cf_suitable_lactose' => 'Suitable for Lactose-Free Diets'
        );

        $current_terms = get_the_terms($id, 'diet');
        if (is_array($current_terms)) {
            foreach ($current_terms as $term) {
                wp_remove_object_terms($id, $term->term_id, 'diet');
            }
        }
        if (array_key_exists('suitable_diets', $item['values'])) {
            $terms = array();
            foreach ($item['values']['suitable_diets'][0]['data'] as $term) {
                if (!term_exists($term, 'diet')) {
                    wp_insert_term($diet[$term], 'diet', ['slug' => $term]);
                }
                array_push($terms, $term);
            }
            wp_set_post_terms($id, $terms, 'diet');
        }

        // Category
        if (array_key_exists('categories', $item)) {
            foreach ($item['categories'] as $slug) {
                if (term_exists($slug, 'product-category')) {
                    $term = get_term_by('slug', $slug, 'product-category');
                    wp_set_object_terms($id, $term->slug, 'product-category');
                }
            }
        }

        // Own Brand QAS
        $this->updateFiles($item, $id);

        // Images
        $this->updateImages($item, $id);
    }

    private function updateTaxTerms($tax, $array, $item, $id, $match = 'yes')
    {
        foreach ($array as $key => $value) {
            if (array_key_exists($key, $item['values'])) {
                if ($item['values'][$key][0]['data'] == $match) {
                    $slug = trim(strtolower($value));
                    if (!term_exists($slug, $tax)) {
                        wp_insert_term($value, $tax, ['slug' => $slug]);
                    }
                    wp_set_object_terms($id, $slug, $tax);
                }
            }
        }
    }

    private function checkForFile($filename)
    {
        global $wpdb;
        $finename = pathinfo($filename);
        $finename = $finename['filename'];
        return intval($wpdb->get_var("SELECT post_id FROM {$wpdb->postmeta} WHERE meta_value LIKE '%$filename%'"));
    }

    public function updateFiles($item, $id)
    {
        require_once('wp-load.php');
        require_once('wp-admin/includes/file.php');
        require_once('wp-admin/includes/media.php');
        require_once('wp-admin/includes/image.php');
        $this->debug('updateFile');

        //
        while (delete_row('product_downloads_pim', 1, $id));
        if (array_key_exists('own_brand_qas', $item['values'])) {

            foreach($item['values']['own_brand_qas'] as $item_def) {
                $this->debug("c");
                $file = $item_def['_links']['download']['href'];

                $basename = $file;
                if (str_ends_with($file, '/download')) {
                    $basename =  substr($file, 0, strlen($file) - strlen('/download'));
                }
                $this->debug($basename);
                if ($this->checkForFile(basename($basename)) == null) {
                    $this->debug("File Does Not Exist");

                    $upload = wp_upload_bits(basename($basename), null, $this->downloadFile($file));
                    $args = array(
                        'guid' => basename($basename),
                        'post_mime_type' => $upload['type'],
                        'post_title' => basename($basename),
                        'post_content' => '',
                        'post_status' => 'inherit'
                    );
                    
                    // Insert the attachment into the database.
                    $attachment_id = wp_insert_attachment($args, $upload['file'], $id);
                    $this->debug($attachment_id);
                } else {
                    $this->debug("File Exists");
                    $attachment_id = $this->checkForFile(basename($basename));
                    $this->debug("fileid: " . $attachment_id);
                }

                $row = array('title' => 'Download QAS Sheet', 'file' => $attachment_id);
                add_row('product_downloads_pim', $row, $id);
            }
        }

        // COSHH
        while (delete_row('product_downloads_coshh', 1, $id));
        if (array_key_exists('own_brand_COSHH', $item['values'])) {

            foreach($item['values']['own_brand_COSHH'] as $item_def) {
                $file = $item_def['_links']['download']['href'];

                $basename = $file;
                if (str_ends_with($file, '/download')) {
                    $basename =  substr($file, 0, strlen($file) - strlen('/download'));
                }
                $this->debug($basename);

                if ($this->checkForFile(basename($basename)) == null) {
                    $this->debug("File Does Not Exist");

                    $upload = wp_upload_bits(basename($basename), null, $this->downloadFile($file));
                    $args = array(
                        'guid' => basename($basename),
                        'post_mime_type' => $upload['type'],
                        'post_title' => basename($basename),
                        'post_content' => '',
                        'post_status' => 'inherit'
                    );
                    
                    // Insert the attachment into the database.
                    $attachment_id = wp_insert_attachment($args, $upload['file'], $id);
                    $this->debug($attachment_id);
                } else {
                    $this->debug("File Exists");
                    $attachment_id = $this->checkForFile(basename($basename));
                }

                $row = array('title' => 'Download COSHH Sheet', 'file' => $attachment_id);
                add_row('product_downloads_coshh', $row, $id);
            }
        }
    }

    public function updateVideoThumbnailImage($item, $id)
    {
        require_once('wp-load.php');
        require_once('wp-admin/includes/file.php');
        require_once('wp-admin/includes/media.php');
        require_once('wp-admin/includes/image.php');

        $this->debug("update Video Thumbnail Image");

        if (array_key_exists('video_thumbnail', $item['values'])) {
            $file = $item['values']['video_thumbnail'][0]['_links']['download']['href'];
            $basename = $file;
            if (str_ends_with($file, '/download')) {
                $basename =  substr($file, 0, strlen($file) - strlen('/download'));
            }

            $this->debug($basename);
            if ($this->checkForFile(basename($basename)) == null) {
                $this->debug("File Does Not Exist");

                $upload = wp_upload_bits(basename($basename), null, $this->downloadFile($file));
                $args = array(
                    'guid' => basename($basename),
                    'post_mime_type' => $upload['type'],
                    'post_title' => basename($basename),
                    'post_content' => '',
                    'post_status' => 'inherit'
                );

                // Insert the attachment into the database.
                $attachment_id = wp_insert_attachment($args, $upload['file'], $id);
                $this->debug($attachment_id);
            } else {
                $this->debug("File Exists");
                $attachment_id = $this->checkForFile(basename($basename));
            }
        }

        if (isset($attachment_id)) {
            update_field('product_video_thumbnail',$attachment_id, $id);
        } else {
            delete_field('product_video_thumbnail', $id);
        }
    }

    public function updateImages($item, $id)
    {
        require_once('wp-load.php');
        require_once('wp-admin/includes/file.php');
        require_once('wp-admin/includes/media.php');
        require_once('wp-admin/includes/image.php');

        $this->debug("update images");

        $keys = array('web_image_1', 'web_image_2', 'web_image_3', 'web_image_4');
        $attachement_ids = array();
        foreach ($keys as $key) {
            if (array_key_exists($key, $item['values'])) {
                $file = $item['values'][$key][0]['_links']['download']['href'];
                $basename = $file;
                if (str_ends_with($file, '/download')) {
                    $basename =  substr($file, 0, strlen($file) - strlen('/download'));
                }

                $this->debug($basename);
                if ($this->checkForFile(basename($basename)) == null) {
                    $this->debug("File Does Not Exist");

                    $upload = wp_upload_bits(basename($basename), null, $this->downloadFile($file));
                    $args = array(
                        'guid' => basename($basename),
                        'post_mime_type' => $upload['type'],
                        'post_title' => basename($basename),
                        'post_content' => '',
                        'post_status' => 'inherit'
                    );
                    
                    // Insert the attachment into the database.
                    $attachment_id = wp_insert_attachment($args, $upload['file'], $id);
                    $this->debug($attachment_id);
                } else {
                    $this->debug("File Exists");
                    $attachment_id = $this->checkForFile(basename($basename));
                }

                array_push($attachement_ids, $attachment_id);
            }
        }

        // Delete all the current rows
        while (delete_row('product_images', 1, $id));

        // Add
        foreach ($attachement_ids as $attach) {
            $row = array('image' => $attach);
            add_row('product_images', $row, $id);
        }
    }
}
