<?php
/*
	Plugin Name: Akeneo PIM
	Plugin URI: https://trunkbbi.com
	Author: TrunkBBI
	Author URI: https://trunkbbi.com
	Description: TrunkBBI custom plugin to fetch data from Akeneo PIM API
	Version: 0.0.5
*/

if (!defined('ABSPATH')) {
    die();
}

if (!class_exists('AkeneoPIM')) {

    class AkeneoPIM
    {
        const AKENEO_BASE_URL = 'https://caterforce.cloud.akeneo.com/';
        const TOKEN_TRANSIENT = 'akeneo_token';
        const REFRESH_TOKEN_TRANSIENT = 'akeneo_refresh_token';

        // Don't do this in real repo!!!
        const CLIENT_ID = '';
        const CLIENT_SECRET = '';
        const CLIENT_USERNAME = '';
        const CLIENT_PASSWORD = '';

        public function __construct()
        {
            $this->initPlugin();
            $this->initAdminPage();

            add_filter("plugin_action_links_" . plugin_basename(__FILE__), function ($links) {
                $settings_link = '<a href="options-general.php?page=akeneo-pim-menu.php">Settings</a>';
                array_unshift($links, $settings_link);
                return $links;
            });

            $this->createPostType();
        }


        private function initPlugin()
        {
            register_activation_hook(__FILE__, [$this, 'activatePlugin']);
            register_deactivation_hook(__FILE__, [$this, 'deactivatePlugin']);
        }

        public function activatePlugin()
        {
        }

        public function deactivatePlugin()
        {
        }

        public function initAdminPage()
        {
            add_action('admin_menu', function () {
                add_submenu_page(null, 'Akeneo PIM', 'Akeneo PIM', 'manage_options', 'akeneo-pim-menu', [$this, 'pluginAdminPage']);
            });
        }

        public function createPostType()
        {
            add_action('init', array($this, 'register_akeneoproduct_type'), 10);
        }

        public function register_akeneoproduct_type()
        {
            if (post_type_exists('members-qas')) {
                return;
            }

            $args = array(
                'label' => 'Member QAS',
                'public' => false,
                'show_ui' => true,
                'menu_icon' => 'dashicons-groups',
            );

            register_post_type('members-qas', $args);
        }

        public function pluginAdminPage()
        {
            echo '<div class="wrap">';
            echo '	<h2>Akeneo PIM</h2>';
            echo '	<form action="' . get_admin_url() . '" method="post">';
            echo '      <input type="hidden" id="trigger_akeneo" name="trigger_akeneo" value="true" />';
            submit_button(__('Fetch Products'));
            echo '	</form>';
            echo '</div>';
        }

        public function fetchAkeneoProducts()
        {
            // fetch auth token for product request
            $akeneo_token = get_transient($this::TOKEN_TRANSIENT);

            if (!$akeneo_token) {
                $akeneo_token = $this->getAkeneoToken();
            }

            if (!$akeneo_token) {
                error_log('failed to fetch akeneo token', 0);
                return;
            }

            // make akeneo product request
            $this->getAkeneoProducts($akeneo_token);
        }

        public function getAkeneoToken()
        {
            $token = false;
            $refresh_token = get_transient($this::REFRESH_TOKEN_TRANSIENT);
            $headers = array(
                'Authorization' => 'Basic ' . base64_encode($this::CLIENT_ID . ':' . $this::CLIENT_SECRET),
                'Content-Type' => 'application/json',
            );

            if (!$refresh_token) {
                // get new set of tokens using pw
                $body = json_encode(array(
                    'username' => $this::CLIENT_USERNAME,
                    'password' => $this::CLIENT_PASSWORD,
                    'grant_type' => 'password',
                ));
            } else {
                // get new token using refresh
                $body = json_encode(array(
                    'refresh_token' => $refresh_token,
                    'grant_type' => 'refresh_token',
                ));
            }

            $args = array(
                'headers' => $headers,
                'body' => $body,
            );
            $response = wp_remote_post($this::AKENEO_BASE_URL . 'api/oauth/v1/token', $args);
            if (!$response) {
                return $token;
            }

            $data = json_decode(wp_remote_retrieve_body($response), true);
            if (!isset($data['access_token'], $data['expires_in'], $data['refresh_token'])) {
                return $token;
            }

            $token = $data['access_token'];
            $expires = $data['expires_in'];
            $refresh_token = $data['refresh_token'];
            set_transient($this::TOKEN_TRANSIENT, $token, $expires);
            set_transient($this::REFRESH_TOKEN_TRANSIENT, $refresh_token, 1209600);
            return $token;
        }

        public function getAkeneoProducts($token)
        {
            $headers = array(
                'Authorization' => 'Bearer ' . $token,
                'Content-Type' => 'application/json',
            );
            $args = array(
                'headers' => $headers,
            );
            $response = wp_remote_get($this::AKENEO_BASE_URL . 'api/rest/v1/products', $args);
            if (!$response) {
                return false;
            }

            $data = json_decode(wp_remote_retrieve_body($response), true);

            if (!isset($data['_embedded']['items']) && is_array($data['_embedded']['items'])) {
                return false;
            }

            $products = [];

            foreach ($data['_embedded']['items'] as $item) {
                if (isset($item['values']['cf_product_name'][0]['data'])) {
                    $product = array(
                        'post_title' => $item['values']['cf_product_name'][0]['data'],
                        'post_content' => '',
                        'post_status' => 'publish',
                        'post_type' => 'members-qas',
                        'post_author' => 1,
                    );
                    $products[] = $product;
                }
            }

            add_action('init', function () use ($products) {
                foreach ($products as $product) {
                    wp_insert_post($product);
                }
            });
        }
    }

    $akeneoPIM = new AkeneoPIM();

    if (isset($_POST['trigger_akeneo']) && "true" === $_POST['trigger_akeneo']) {
        $akeneoPIM->fetchAkeneoProducts();
    }
}
