<?php

namespace Tbbi\CF;

use WP_Query;
use WP_Error;

class CaterforceAuth
{
    private const AKENEO_BASE_URL = 'https://caterforce.cloud.akeneo.com/';
    private const AKENEO_MEDIA_URL = 'https://caterforce.cloud.akeneo.com/media/download/';
    private const AKENEO_AUTH = 'api/oauth/v1/token';
    private const AKENO_BRANDS = 'api/rest/v1/reference-entities/brand_entity/records?limit=100';
    private const AKENEO_PRODUCTS = 'api/rest/v1/products?limit=25&search=';
    private const ADMIN_PAGE = 'options-general.php?page=akeneo-pim-menu.php';
    private const AKENO_BRANDS_SEARCH = '{"cf_brand":[{"operator": "IN","value": [%BRANDS%]}]}';

    private $client_id;
    private $client_secret;
    private $client_refresh;
    private $client_token;

    public $brands_array;

    public function __construct()
    {
        $this->client_id = get_option('cf-id');
        $this->client_secret = get_option('cf-secret');
        $this->client_refresh = get_option('cf-refresh');
        $this->client_token = get_option('cf-token');

        add_action('admin_init', function () {
            $this->brands_array = $this->getBrandsArray();
        });

        add_action('admin_post_get_cf_token', [$this, 'getOauthToken']);
        add_action('admin_post_renew_cf_token', [$this, 'renewOauthToken']);
        add_action('admin_post_destroy_cf_token', [$this, 'destroyOauthToken']);

        add_action('admin_post_schedule_import_brands', [$this, 'scheduleBrandsImport'], 10, 1);
        add_action('admin_post_schedule_import', [$this, 'scheduleProductsImport'], 10, 1);
        add_action('admin_post_schedule_import_now', [$this, 'scheduleProductsImport'], 10, 1);
        add_action('admin_post_set_brands', [$this, 'setBrands']);

        add_action('cf_import_brands', [$this, 'getBrands'], 10, 1);
        add_action('cf_import_products', [$this, 'getProducts'], 10, 1);
        add_action('cf_schedule_import_products', [$this, 'getProducts'], 10, 1);
    }

    ////////////////////////////////////
    //// AUTH
    ////////////////////////////////////
    private function makeRequest($body, $endpoint)
    {
        error_log("makeRequest: " . $endpoint);
        $headers = array(
            'Authorization' => 'Basic ' . base64_encode($this->client_id . ':' . $this->client_secret),
            'Content-Type' => 'application/json',
        );

        $args = array(
            'headers' => $headers,
            'body' => $body
        );

        return wp_remote_post($this::AKENEO_BASE_URL . $endpoint, $args);
    }

    /**
     * Creates the initial oauth token
     */
    public function getOauthToken()
    {
        error_log("getOauthToken");
        $redirect = false;
        if (isset($_POST['redirect']) && $_POST['redirect'] == 'true') {
            $redirect = true;
        }

        if (!isset($_POST['cf-username']) || !isset($_POST['cf-password'])) {
            if ($redirect) {
                wp_redirect(admin_url($this::ADMIN_PAGE));
            }
            die();
        }

        $username = $_POST['cf-username'];
        $password = $_POST['cf-password'];

        update_option('cf-id', $_POST['cf-id']);
        update_option('cf-secret', $_POST['cf-secret']);

        $this->client_id = get_option('cf-id');
        $this->client_secret = get_option('cf-secret');

        $body = json_encode(array(
            'username' => $username,
            'password' => $password,
            'grant_type' => 'password',
        ));
        $response = $this->makeRequest($body, $this::AKENEO_AUTH);

        if (!$response) {
            if ($redirect) {
                wp_redirect(admin_url($this::ADMIN_PAGE));
            }
            die();
        }

        $data = json_decode(wp_remote_retrieve_body($response), true);
        if (!isset($data['access_token'], $data['expires_in'], $data['refresh_token'])) {
            if ($redirect) {
                wp_redirect(admin_url($this::ADMIN_PAGE));
            }
            die();
        }

        update_option('cf-token', $data['access_token']);
        update_option('cf-refresh', $data['refresh_token']);

        // Front end
        if ($redirect) {
            wp_redirect(admin_url($this::ADMIN_PAGE));
            die();
        }
    }

    /**
     * Renews the oauth token
     */
    public function renewOauthToken()
    {
        error_log("renewOauthToken");
        $redirect = false;
        if (isset($_POST['redirect']) && $_POST['redirect'] == 'true') {
            $redirect = true;
        }

        $body = json_encode(array(
            'refresh_token' => $this->client_refresh,
            'grant_type' => 'refresh_token',
        ));

        $response = $this->makeRequest($body, $this::AKENEO_AUTH);

        if (!$response) {
            if ($redirect) {
                wp_redirect($this::ADMIN_PAGE);
            }
            die();
        }

        $data = json_decode(wp_remote_retrieve_body($response), true);

        if (!isset($data['access_token'], $data['expires_in'], $data['refresh_token'])) {
            if ($redirect) {
                wp_redirect(admin_url($this::ADMIN_PAGE));
            }
            die();
        }

        update_option('cf-token', $data['access_token']);
        update_option('cf-refresh', $data['refresh_token']);

        $this->client_refresh = get_option('cf-refresh');
        $this->client_token = get_option('cf-token');

        // Front end
        if ($redirect) {
            wp_redirect(admin_url($this::ADMIN_PAGE));
            die();
        }
    }

    /**
     * Deletes all oauth tokens and secrets
     */
    public function destroyOauthToken()
    {
        $redirect = false;
        if (isset($_POST['redirect']) && $_POST['redirect'] == 'true') {
            $redirect = true;
        }

        update_option('cf-id', null);
        update_option('cf-secret', null);
        update_option('cf-token', null);
        update_option('cf-refresh', null);

        // Front end
        if ($redirect) {
            wp_redirect(admin_url('options-general.php?page=akeneo-pim-menu.php'));
            die();
        }
    }

    ////////////////////////////////////
    ////  BRANDS
    ////////////////////////////////////
    public function scheduleBrandsImport($url = '')
    {
        $timestamp = strtotime(date("Y-m-d H:i:s", time() + 10)); // runs 10 sec from now
        if ($url !== '') {
            wp_schedule_single_event($timestamp, 'cf_import_brands', array($url), true);
        } else {
            $url = $this::AKENEO_BASE_URL . $this::AKENO_BRANDS;
            wp_schedule_single_event($timestamp, 'cf_import_brands', array($url), true);
        }
        wp_redirect(admin_url($this::ADMIN_PAGE));
        die();
    }

    /**
     * Gets all the brands from Akeno PIM
     */
    public function getBrands($url)
    {
        error_log("getBrands");
        $args = array(
            'headers' => array(
                'Authorization' => 'Bearer ' . $this->client_token,
                'Content-Type' => 'application/json',
            )
        );

        $response = wp_remote_get($url, $args);
        if (!$response) {
            return false;
        }

        if (wp_remote_retrieve_response_code($response) != 200) {
            $this->renewOauthToken();
            $this->scheduleBrandsImport($url);
            die();
        }

        $data = json_decode(wp_remote_retrieve_body($response), true);
        if (!isset($data['_embedded']['items']) && is_array($data['_embedded']['items'])) {
            return false;
        }

        foreach ($data['_embedded']['items'] as $brand) {
            $slug = $brand['code'];
            $value = $brand['values']['label'][0]['data'];

            if (!term_exists($slug, 'brands')) {
                wp_insert_term($value, 'brands', ['slug' => $slug]);
            }
        }

        if (array_key_exists('next', $data['_links'])) {
            $this->scheduleBrandsImport($data['_links']['next']['href']);
        } else {
            die();
        }
    }

    /**
     * Gets list of brands for use
     *
     * @return void
     */
    public function getBrandsArray()
    {
        $brands_required = '';
        $brands = get_terms([
            'taxonomy' => 'brands',
            'hide_empty' => false,
        ]);
        foreach ($brands as $brand) {
            if (get_field('brand_used', 'brands_' . $brand->term_id)) {
                $brands_required .= '"' . $brand->slug . '",';
            }
        }
        $brands_required = rtrim($brands_required, ',');

        return str_replace("%BRANDS%", $brands_required, $this::AKENO_BRANDS_SEARCH);
    }

    /**
     * Sets the brands in use
     */
    public function setBrands()
    {
        $brands = get_terms([
            'taxonomy' => 'brands',
            'hide_empty' => false,
        ]);

        foreach ($brands as $brand) {
            update_field('brand_used', false, 'brands_' . $brand->term_id);
        }

        foreach ($_POST as $key => $value) {
            if (substr($key, 0, 9) === "cf-brand-") {
                $id = preg_replace('/[^0-9]/', '', $key);
                update_field('brand_used', true, 'brands_' . $id);
            }
        }
        wp_redirect(admin_url($this::ADMIN_PAGE));
    }

    ////////////////////////////////////
    // PRODUCTS
    ////////////////////////////////////
    public function scheduleProductsImport($url = '')
    {
        error_log("scheduleProductsImport");
        if (isset($_POST['time']) && $_POST['time']) {
            $time = $_POST['cf-time'];
            update_option('cf-schedule', $time);
            wp_clear_scheduled_hook('cf_schedule_import_products');

            if (!wp_next_scheduled('cf_schedule_import_products', $args)) {
                $url = $this::AKENEO_BASE_URL . $this::AKENEO_PRODUCTS . $this->brands_array;
                wp_schedule_event(strtotime($time), 'daily', 'cf_schedule_import_products', array($url), true);
            }
        } else {
            $timestamp = strtotime(date("Y-m-d H:i:s", time() + 10));
            if ($url !== '') {
                wp_schedule_single_event($timestamp, 'cf_import_products', array($url), true);
            } else {
                $url = $this::AKENEO_BASE_URL . $this::AKENEO_PRODUCTS . $this->brands_array;
                wp_schedule_single_event($timestamp, 'cf_import_products', array($url), true);
            }
            wp_redirect(admin_url($this::ADMIN_PAGE));
            die();
        }
    }


    public function getProducts($url)
    {
        error_log("getProducts");
        $args = array(
            'headers' => array(
                'Authorization' => 'Bearer ' . $this->client_token,
                'Content-Type' => 'application/json',
            )
        );

        $response = wp_remote_get($url, $args);
        if (!$response) {
            return false;
        }

        if (wp_remote_retrieve_response_code($response) != 200) {
            $this->renewOauthToken();
            $this->scheduleProductsImport($url);
            die();
        }

        // Get the content
        $data = json_decode(wp_remote_retrieve_body($response), true);
        if (!isset($data['_embedded']['items']) && is_array($data['_embedded']['items'])) {
            return false;
        }

        // parse
        $this->parseProducts($data['_embedded']['items']);

        // Get next
        if (array_key_exists('next', $data['_links'])) {
            error_log("get next");
            $this->scheduleProductsImport($data['_links']['next']['href']);
        } else {
            error_log("complete");
            die();
        }
    }

    private function deleteProduct()
    {
    }

    private function parseProducts($data)
    {
        error_log("parseProducts");
        $prods = array();

        $query = new WP_Query(array(
            'posts_per_page' => -1,
            'fields' => 'ids',
            'post_type' => 'qas'
        ));
        if ($query->have_posts()) {
            while ($query->have_posts()) {
                $query->the_post();
                $id = get_the_ID();
                $prods[$id] =  get_field('product_identifier', $id);
            }
        }
        // Loop each product to see if it exists, matches on product_identifier
        foreach ($data as $item) {
            if (in_array($item['identifier'], $prods)) {
                error_log("Update:");
                $id = array_search($item['identifier'], $prods);
            } else {
                error_log("New:");
                $id = $this->createProduct($item);
            }
            error_log("ID: " . $id);
            $this->updateProduct($id, $item);
        }
    }

    private function createProduct($item)
    {
        error_log("createProduct");
        $post = array(
            'post_type' => 'qas',
            'post_content' => '',
            'post_excerpt' => '',
            'post_status' => 'publish',
            'post_title' => $item['values']['cf_product_name'][0]['data'],
            'post_name' => '',
            'post_date' => $item['created'],
            'post_modified' => $item['updated'],
            'comment_status' => 'closed',
            'post_author' => 1
        );
        $id = wp_insert_post($post, true);
        if (is_wp_error($id)) {
            $errors = $id->get_error_messages();
            foreach ($errors as $error) {
                error_log($error);
            }
            die();
        } else {
            return $id;
        }
    }

    private function updateProduct($id, $item)
    {
        error_log("update: " . $item['values']['cf_product_name'][0]['data']);

        // Identifier
        update_field('product_identifier', $item['identifier'], $id);

        // General
        if (array_key_exists('erudus_traded_unit_gtin', $item['values'])) {
            update_field('product_gtin', $item['values']['erudus_traded_unit_gtin'][0]['data'], $id);
        }

        if (array_key_exists('cf_csbc', $item['values'])) {
            update_field('product_csbc', $item['values']['cf_csbc'][0]['data'], $id);
        }

        if (array_key_exists('erudus_id', $item['values'])) {
            update_field('erudus_id', $item['values']['erudus_id'][0]['data'], $id);
        }

        if (array_key_exists('erudus_status', $item['values'])) {
            update_field('erudus_status', $item['values']['erudus_status'][0]['data'], $id);
        }

        if (array_key_exists('cf_country_of_origin', $item['values'])) {
            update_field('country_of_origin', $item['values']['cf_country_of_origin'][0]['data'], $id);
        }

        if (array_key_exists('cf_product_description', $item['values'])) {
            update_field('product_overview', $item['values']['cf_product_description'][0]['data'], $id);
        }

        if (array_key_exists('erudus_suppliers_product_code', $item['values'])) {
            update_field('product_supplier_code', $item['values']['erudus_suppliers_product_code'][0]['data'], $id);
        }

        if (array_key_exists('erudus_supplier_name', $item['values'])) {
            update_field('product_supplier', $item['values']['erudus_supplier_name'][0]['data'], $id);
        }

        if (array_key_exists('erudus_storage_instructions', $item['values'])) {
            update_field('product_storage', $item['values']['erudus_storage_instructions'][0]['data'], $id);
        }

        if (array_key_exists('erudus_directions_for_use', $item['values'])) {
            update_field('product_directions', $item['values']['erudus_directions_for_use'][0]['data'], $id);
        }

        if (array_key_exists('erudus_ingredients', $item['values'])) {
            update_field('product_ingredients', $item['values']['erudus_ingredients'][0]['data'], $id);
        }

        // Brand
        if (array_key_exists('cf_brand', $item['values'])) {
            $brand_slug = $item['values']['cf_brand'][0]['data'];
            wp_set_object_terms($id, $brand_slug, 'brands');
        }

        // Allergens
        $allergens = array(
            'cf_contains_sulphur_sulphites' => 'Sulphites',
            'cf_contains_soybeans' => 'Soybeans',
            'cf_contains_crustacea' => 'Crustaea',
            'cf_contains_molluscs' => 'Molluscs',
            'cf_contains_eggs' => 'Eggs',
            'cf_contains_nuts' => 'Nuts',
            'cf_contains_mustard' => 'Mustard',
            'cf_contains_cereals' => 'Cereals',
            'cf_contains_peanuts' => 'Peanuts',
            'cf_contains_milk' => 'Milk',
            'cf_contains_lupin' => 'Lupin',
            'cf_contains_sesame' => 'Seasame',
            'cf_contains_fish' => 'Fish',
            'cf_celery_celeriac' => 'Celerey'
        );
        $this->updateTaxTerms('allergens', $allergens, $item, $id);

        // Diet
        $diet = array(
            'cf_halal_diet' => 'Halal',
            'cf_kosher_diet' => 'Kosher',
            'cf_suitable_vegan' => 'Vegan',
            'cf_suitable_vegetarian' => 'Vegetarian',
            'cf_suitable_coeliacs' => 'Coeliac',
            'cf_suitable_lactose' => 'Lactose'
        );
        $this->updateTaxTerms('diet', $diet, $item, $id);

        // Category
        if (array_key_exists('erudus_product_category', $item['values'])) {
            $category = $item['values']['erudus_product_category'][0]['data'];
            $cat_slug = trim(strtolower($category));
            if (!term_exists($cat_slug, 'product-category')) {
                wp_insert_term($category, 'product-category', ['slug' => $cat_slug]);
            }
            wp_set_object_terms($id, $cat_slug, 'product-category');
        }

        // Images
        $this->updateImages($item, $id);
    }

    private function updateTaxTerms($tax, $array, $item, $id)
    {
        foreach ($array as $key => $value) {
            if (array_key_exists($key, $item['values'])) {
                if ($item['values'][$key][0]['data'] == 'yes') {
                    $slug = trim(strtolower($value));
                    if (!term_exists($slug, $tax)) {
                        wp_insert_term($value, $tax, ['slug' => $slug]);
                    }
                    wp_set_object_terms($id, $slug, $tax);
                }
            }
        }
    }

    private function checkForFile($filename)
    {
        global $wpdb;
        return intval($wpdb->get_var("SELECT post_id FROM {$wpdb->postmeta} WHERE meta_value LIKE '%/$filename'"));
    }

    public function updateImages($item, $id)
    {
        // TODO, check these...
        require_once('wp-load.php');
        require_once('wp-admin/includes/file.php');
        require_once('wp-admin/includes/media.php');
        require_once('wp-admin/includes/image.php');

        $admin_path = str_replace(get_bloginfo('url') . '/', ABSPATH, get_admin_url());
        error_log("update images");

        $keys = array('web_image_1', 'web_image_2', 'web_image_3', 'web_image_4');
        $attachement_ids = array();
        foreach ($keys as $key) {
            if (array_key_exists($key, $item['values'])) {
                // Because the Akeneo PIM system does a double encode on the slashes in the filename...
                $url = str_replace("/", "%252F", $item['values'][$key][0]['data']);
                $url = $this::AKENEO_MEDIA_URL . $url;

                if ($this->checkForFile(basename($url)) == null) {
                    $tmp_file = array();
                    $tmp_file['name'] = basename($url);
                    $tmp_file['tmp_name'] = download_url($url);
                    $attachment_id = media_handle_sideload($tmp_file);
// Todo - error checking
                } else {
                    $attachment_id = $this->checkForFile(basename($url));
                }

                // Add to arr
                array_push($attachement_ids, $attachment_id);
            }
        }

        // Delete all the current rows
        $images = get_field('product_images', $id);
        if (is_array($images)) {
            $i = 1;
            foreach ($images as $image) {
                delete_row('images', $i);
                $i++;
            }
        }

        // Add
        foreach ($attachement_ids as $attach) {
            $row = array('image' => $attach);
            add_row('product_images', $row, $id);
        }
    }
}
