<?php
/**
 * Plugin Name:     Trunk BBI Security
 * Plugin URI:      https://trunkbbi.com
 * Description:     Simple WordPress plugin to harden WordPress
 * Author:          R.Jones-Smith
 * Author URI:      https://trunkbbi.com
 * Text Domain:     trunk-bbi-security
 * Domain Path:     /languages
 * Version:         0.1.0
 *
 * @package         Trunk_Bbi_Security
 */

if (!defined('ABSPATH')) {
    die();
}

if (!class_exists('TrunkBBI')) {

    class TrunkBBI
    {
        public function __construct()
        {
            $this->initPlugin();
            $this->initAdminPage();

            add_filter("plugin_action_links_" . plugin_basename(__FILE__), function ($links) {
                $settings_link = '<a href="options-general.php?page=trunk-bbi-security.php">Settings</a>';
                array_unshift($links, $settings_link);
                return $links;
            });

            $this->disableJquery();
            $this->disableVersionNumbers();
            $this->disableComments();
            $this->disableXmlrpc();
            $this->disableACF();
            $this->disableEmojis();
            $this->disableGutenberg();
            $this->disableFeeds();
            $this->disableRest();
            $this->disableAuthorPages();
            $this->disableAdminSections();
            $this->disableAdminMenuItems();
            $this->disableUserOptions();
            $this->disableLoginErrors();
            $this->disablePluginActivate();
            $this->disablePluginDeactivate();
        }


        private function initPlugin()
        {
            register_activation_hook(__FILE__, [$this, 'activatePlugin']);
            register_deactivation_hook(__FILE__, [$this, 'deactivatePlugin']);
        }

        public function activatePlugin()
        {
        }

        public function deactivatePlugin()
        {
        }

        public function initAdminPage()
        {
            add_action('admin_init', function () {
                add_settings_section('tbbi_security_settings_section', '', false, 'tbbi_security_settings');
                $fields = array(
                    array('uid' => 'tbbi_disable_jquery', 'label' => 'Disable jQuery', 'section' => 'tbbi_security_settings_section', 'type' => 'checkbox', 'options' => false, 'supplemental' => 'Prevents jQuery from being loaded on the front-end of the website.'),
                    array('uid' => 'tbbi_disable_versions', 'label' => 'Disable WordPress version numbers', 'section' => 'tbbi_security_settings_section', 'type' => 'checkbox', 'options' => false, 'supplemental' => 'Removes or nullifies the WordPress version number in various places.'),
                    array('uid' => 'tbbi_disable_comments', 'label' => 'Disable comments', 'section' => 'tbbi_security_settings_section', 'type' => 'checkbox', 'options' => false, 'supplemental' => 'Prevents comments and pingbacks from being left.'),
                    array('uid' => 'tbbi_disable_emojis', 'label' => 'Disable emoji support', 'section' => 'tbbi_security_settings_section', 'type' => 'checkbox', 'options' => false, 'supplemental' => 'Disables WordPress\' emoji scripts'),
                    array('uid' => 'tbbi_disable_xmlrpc', 'label' => 'Disable XMLRPC', 'section' => 'tbbi_security_settings_section', 'type' => 'checkbox', 'options' => false, 'supplemental' => 'Disables XMLRPC'),
                    array('uid' => 'tbbi_disable_gutenberg', 'label' => 'Disable Gutenberg block editor', 'section' => 'tbbi_security_settings_section', 'type' => 'checkbox', 'options' => false, 'supplemental' => 'Forces the standard text edititor'),
                    array('uid' => 'tbbi_disable_feeds', 'label' => 'Disable feeds', 'section' => 'tbbi_security_settings_section', 'type' => 'checkbox', 'options' => false, 'supplemental' => 'Disables all RSS style feeds'),
                    array('uid' => 'tbbi_disable_rest', 'label' => 'Disable REST API', 'section' => 'tbbi_security_settings_section', 'type' => 'checkbox', 'options' => false, 'supplemental' => 'Disables the WordPress JSON API'),
                    array('uid' => 'tbbi_disable_authors', 'label' => 'Disable author pages', 'section' => 'tbbi_security_settings_section', 'type' => 'checkbox', 'options' => false, 'supplemental' => 'Blocks access to author pages'),
                    array('uid' => 'tbbi_disable_acf', 'label' => 'Block access to ACF', 'section' => 'tbbi_security_settings_section', 'type' => 'checkbox', 'options' => false, 'supplemental' => 'Blocks access to ACF settings'),
                    array('uid' => 'tbbi_disable_admin_sections', 'label' => 'Disable admin sections', 'section' => 'tbbi_security_settings_section', 'type' => 'checkbox', 'options' => false, 'supplemental' => 'Hides various admin areas from all users'),
                    array('uid' => 'tbbi_disable_admin_menus', 'label' => 'Disable admin menu items', 'section' => 'tbbi_security_settings_section', 'type' => 'checkbox', 'options' => false, 'supplemental' => 'Hides various admin menu items from all users'),
                    array('uid' => 'tbbi_disable_admin_login_errors', 'label' => 'Disable admin login errors', 'section' => 'tbbi_security_settings_section', 'type' => 'checkbox', 'options' => false, 'supplemental' => 'Forces all admin login errors to have a non descriptive reason for failure'),
                    array('uid' => 'tbbi_disable_user_profile', 'label' => 'Disable user profile options', 'section' => 'tbbi_security_settings_section', 'type' => 'checkbox', 'options' => false, 'supplemental' => 'Hides various unneccesary user profile elements'),
                    array('uid' => 'tbbi_disable_plugin_activate', 'label' => 'Disable plugin activate', 'section' => 'tbbi_security_settings_section', 'type' => 'checkbox', 'options' => false, 'supplemental' => 'Prevents plugins from being activated through the WordPress admin'),
                    array('uid' => 'tbbi_disable_plugin_deactivate', 'label' => 'Disable plugin deactivate', 'section' => 'tbbi_security_settings_section', 'type' => 'checkbox', 'options' => false, 'supplemental' => 'Prevents plugins from being deactivated through the WordPress admin')
                );

                foreach ($fields as $field) {
                    add_settings_field($field['uid'], $field['label'], [$this, 'displaySettingsField'], 'tbbi_security_settings', $field['section'], $field);
                    register_setting('tbbi_security_settings', $field['uid']);
                }
            });

            add_action('admin_menu', function () {
                add_submenu_page(null, 'TrunkBBI Security', 'TrunkBBI Security', 'manage_options', 'trunk-bbi-security', [$this, 'pluginAdminPage']);
            });
        }

        public function pluginAdminPage()
        {
            echo '<div class="wrap">';
            echo '	<h2>TBBI Security</h2>';
            echo '	<form action="options.php" method="post">';
            echo '		<table class="form-table">';

            settings_fields('tbbi_security_settings');
            do_settings_sections('tbbi_security_settings');

            echo    '	</table>';

            submit_button(__('Save Settings', 'tbbi'));

            echo '	</form>';
            echo '</div>';
        }

        public function displaySettingsField($args)
        {
            $value = get_option($args['uid']);
            if ($value) {
                echo '<input name="' . $args['uid'] . '" id="' . $args['uid'] . '" type="checkbox" value="1" checked>';
            } else {
                echo '<input name="' . $args['uid'] . '" id="' . $args['uid'] . '" type="checkbox" value="1">';
            }

            if ($supplimental = $args['supplemental']) {
                printf('<p class="description">%s</p>', $supplimental);
            }
        }

        private function disableACF()
        {
            $disable = get_option('tbbi_disable_acf');
            if ($disable) {
                add_action('admin_menu', function () {
                    remove_menu_page('edit.php?post_type=acf-field-group');
                });
            }
        }


        private function disablePluginDeactivate()
        {
            $disable = get_option('tbbi_disable_plugin_deactivate');
            if ($disable) {
                add_filter('plugin_action_links', function ($actions, $plugin_file, $plugin_data, $context) {
                    if (array_key_exists('deactivate', $actions)) {
                        unset($actions['deactivate']);
                    }
                    return $actions;
                }, 10, 4);
            }
        }

        private function disablePluginActivate()
        {
            $disable = get_option('tbbi_disable_plugin_activate');
            if ($disable) {
                add_filter('plugin_action_links', function ($actions, $plugin_file, $plugin_data, $context) {
                    if (array_key_exists('activate', $actions)) {
                        unset($actions['activate']);
                    }
                    return $actions;
                }, 10, 4);
            }
        }

        private function disableLoginErrors()
        {
            $disable = get_option('tbbi_disable_admin_login_errors');
            if ($disable) {
                add_filter('login_errors', function () {
                    return 'Login unsuccessful';
                });
            }
        }

        private function disableUserOptions()
        {
            $disable = get_option('tbbi_disable_user_profile');
            if ($disable) {
                add_action('admin_head', function () {
                    remove_action('admin_color_scheme_picker', 'admin_color_scheme_picker');
                });
                add_filter('wp_is_application_passwords_available', '__return_false');
                add_action('admin_head-user-edit.php', [$this, 'hideUserFields']);
                add_action('admin_head-profile.php', [$this, 'hideUserFields']);
                add_filter('pre_get_avatar', '__return_false', 10, 3);
            }
        }

        public function hideUserFields()
        {
            echo '<style>
			tr.user-url-wrap,
			tr.user-profile-picture,
			tr.user-description-wrap, #application-passwords-section,
			tr.user-admin-bar-front-wrap,
			tr.user-rich-editing-wrap,
			tr.user-syntax-highlighting-wrap,
			tr.user-comment-shortcuts-wrap,
			#your-profile h2
			{ display: none; }</style>';
        }

        /**
         * Removes various sections
         *
         * @return void
         */
        private function disableAdminSections()
        {
            $disable = get_option('tbbi_disable_admin_sections');
            if ($disable) {
                add_filter('admin_init', function () {
                    remove_meta_box('dashboard_primary', 'dashboard', 'side');
                    remove_meta_box('dashboard_incoming_links', 'dashboard', 'normal');
                    remove_meta_box('dashboard_secondary', 'dashboard', 'side');
                    remove_meta_box('dashboard_quick_press', 'dashboard', 'side');
                    remove_meta_box('dashboard_site_health', 'dashboard', 'normal');
                    remove_action('welcome_panel', 'wp_welcome_panel');
                });
                add_filter('manage_users_columns', function ($columns) {
                    unset($columns['posts']);
                    return $columns;
                });
                add_filter('contextual_help_list', function () {
                    global $current_screen;
                    $current_screen->remove_help_tabs();
                });
                add_action('wp_before_admin_bar_render', function () {
                    global $wp_admin_bar;
                    $wp_admin_bar->remove_menu('updates');
                    $wp_admin_bar->remove_menu('themes');
                });
            }
        }

        /**
         * Hides various admin menu items
         *
         * @return void
         */
        public function disableAdminMenuItems()
        {
            $disable = get_option('tbbi_disable_admin_menus');
            if ($disable) {
                add_action('admin_head', function () {
                    remove_action('admin_notices', 'update_nag', 3);
                    remove_action('admin_notices', 'maintenance_nag', 10);
                    remove_submenu_page('tools.php', 'site-health.php');
                    remove_submenu_page('plugins.php', 'plugin-editor.php');
                    remove_submenu_page('plugins.php', 'plugin-install.php');
                    remove_submenu_page('index.php', 'update-core.php');
                    remove_submenu_page('themes.php', 'theme-editor.php');
                }, 99);
            }
        }

        /**
         * Removes jQuery from the front end
         *
         * @return void
         */
        private function disableJquery()
        {
            $disable = get_option('tbbi_disable_jquery');
            if ($disable) {
                add_action('wp_enqueue_scripts', function () {
                    if (!is_admin()) {
                        wp_deregister_script('jquery');
                    }
                }, 100);
            }
        }

        /**
         * Hide or remove all version numbers
         *
         * @return void
         */
        private function disableVersionNumbers()
        {
            $disable = get_option('tbbi_disable_versions');
            if ($disable) {
                remove_action('wp_head', 'wp_generator');
                add_filter('the_generator', '__return_null');
                add_filter('style_loader_src', [$this, 'removeVersionStrings']);
                add_filter('script_loader_src', [$this, 'removeVersionStrings']);
                add_action('admin_menu', function () {
                    remove_filter('update_footer', 'core_update_footer');
                }, 99);
                add_action('admin_head', function () {
                    echo '<style>#wp-version-message {display: none;}</style>';
                });
            }
        }

        /**
         * Remove version string from styles and scripts
         *
         * @param [type] $src
         * @return void
         */
        public function removeVersionStrings($src)
        {
            if (strpos($src, 'ver=' . get_bloginfo('version'))) {
                $src = remove_query_arg('ver', $src);
            }
            return $src;
        }

        /**
         * Fully block and disable comments for all posts and pages
         *
         *
         * @return void
         */
        private function disableComments()
        {
            $disable = get_option('tbbi_disable_comments');
            if ($disable) {
                add_action('wp_headers', [$this, 'disablePingbacks']);
                add_filter('comments_open', '__return_false', 20, 2);
                add_filter('pings_open', '__return_false', 20, 2);
                add_filter('comments_array', '__return_empty_array', 10, 2);
                add_filter('feed_links_show_comments_feed', '__return_false');
                add_filter('get_comments_number', [$this, 'disableCommentsZeroAll']);
                add_filter('admin_init', [$this, 'disableCommentsAdmin']);
                add_action('init', [$this, 'disableCommentsAdminMenuBar']);
                add_action('wp_before_admin_bar_render', [$this, 'disableCommentsAdminBarIcon']);
                add_action('admin_menu', [$this, 'disableCommentsAdminMenu']);
                add_filter('post_comments_feed_link', function () {
                    return null;
                });
            }
        }

        public function disablePingbacks($headers)
        {
            unset($headers['X-Pingback']);
            return $headers;
        }

        /**
         * Returns zero for all comment counts
         *
         * @param [int] $count
         * @return void
         */
        public function disableCommentsZeroAll($count)
        {
            return 0;
        }

        /**
         * Blocks admin pages and elements related to comments
         *
         * @return void
         */
        public function disableCommentsAdmin()
        {
            global $pagenow;

            if ($pagenow === 'edit-comments.php' || $pagenow === 'options-discussion.php') {
                wp_redirect(admin_url());
                die();
            }

            remove_meta_box('dashboard_recent_comments', 'dashboard', 'normal');
            foreach (get_post_types() as $post_type) {
                if (post_type_supports($post_type, 'comments')) {
                    remove_post_type_support($post_type, 'comments');
                    remove_post_type_support($post_type, 'trackbacks');
                }
            }
        }

        /**
         * Removes the comments menu from the admin bar
         *
         * @return void
         */
        public function disableCommentsAdminMenuBar()
        {
            if (is_admin_bar_showing()) {
                remove_action('admin_bar_menu', 'wp_admin_bar_comments_menu', 60);
            }
        }

        /**
         * Removes the comments icon from the drop down admin bar
         *
         * @return void
         */
        public function disableCommentsAdminBarIcon()
        {
            global $wp_admin_bar;
            $wp_admin_bar->remove_menu('comments');
        }

        /**
         * Removes comments from admin side menu
         *
         * @return void
         */
        public function disableCommentsAdminMenu()
        {
            remove_menu_page('edit-comments.php');
            remove_submenu_page('options-general.php', 'options-discussion.php');
        }


        /**
         * Disable XMLRPC access
         * Note that the server config should also block xmlrpc.php
         *
         * @return void
         */
        private function disableXmlrpc()
        {
            $disable = get_option('tbbi_disable_xmlrpc');
            if ($disable) {
                add_filter('xmlrpc_enabled', '__return_false');
            }
        }

        /**
         * Fully disable emoji support in WordPress
         *
         * @return void
         */
        private function disableEmojis()
        {
            $disable = get_option('tbbi_disable_emojis');
            if ($disable) {
                remove_action('admin_print_scripts', 'print_emoji_detection_script');
                remove_action('admin_print_styles', 'print_emoji_styles');
                remove_action('wp_head', 'print_emoji_detection_script', 7);
                remove_filter('wp_mail', 'wp_staticize_emoji_for_email');
                remove_action('wp_print_styles', 'print_emoji_styles');
                remove_filter('the_content_feed', 'wp_staticize_emoji');
                remove_filter('comment_text_rss', 'wp_staticize_emoji');
                remove_filter('wp_mail', 'wp_staticize_emoji_for_email');
                add_filter('emoji_svg_url', '__return_false');
            }
        }

        /**
         * Fully disable the Gutenberg block editor allowing only the classic editor
         *
         * @return void
         */
        private function disableGutenberg()
        {
            $disable = get_option('tbbi_disable_gutenberg');
            if ($disable) {
                add_filter('use_block_editor_for_post', '__return_false');
                add_filter('use_widgets_block_editor', '__return_false');

                add_action('wp_enqueue_scripts', function () {
                    wp_dequeue_style('wp-block-library');
                    wp_dequeue_style('wp-block-library-theme');
                    wp_dequeue_style('global-styles');
                }, 20);

                add_action('after_setup_theme', function () {
                    remove_theme_support('block-templates');
                    remove_theme_support('core-block-patterns');
                }, 20);
            }
        }

        /**
         * Block and disable RSS and other style feeds
         *
         * @return void
         */
        private function disableFeeds()
        {
            $disable = get_option('tbbi_disable_feeds');
            if ($disable) {
                add_action('do_feed', [$this, 'disableFeedsResponse'], 1);
                add_action('do_feed_rdf', [$this, 'disableFeedsResponse'], 1);
                add_action('do_feed_rss', [$this, 'disableFeedsResponse'], 1);
                add_action('do_feed_rss2', [$this, 'disableFeedsResponse'], 1);
                add_action('do_feed_atom', [$this, 'disableFeedsResponse'], 1);
                add_action('do_feed_rss2_comments', [$this, 'disableFeedsResponse'], 1);
                add_action('do_feed_atom_comments', [$this, 'disableFeedsResponse'], 1);

                remove_action('wp_head', 'wlwmanifest_link');
                remove_action('wp_head', 'feed_links_extra', 3);
                remove_action('wp_head', 'feed_links', 2);
                remove_action('wp_head', 'rsd_link');
                remove_action('wp_head', 'wp_shortlink_wp_head', 10, 0);
                remove_action('wp_head', 'rest_output_link_wp_head', 10);
                remove_action('template_redirect', 'rest_output_link_header', 11);
            }
        }

        /**
         * Force a generaic response into all feeds
         *
         * @return void
         */
        private function disableFeedsResponse()
        {
            wp_die(__('Unauthorised'));
        }

        /**
         * Block the WP REST API
         *
         * @return void
         */
        private function disableRest()
        {
            $disable = get_option('tbbi_disable_rest');
            if ($disable) {
                remove_action('wp_head', 'rest_output_link_wp_head', 10);
                remove_action('wp_head', 'wp_oembed_add_discovery_links', 10);
                remove_action('rest_api_init', 'wp_oembed_register_route');
                remove_filter('oembed_dataparse', 'wp_filter_oembed_result', 10);
                remove_action('wp_head', 'wp_oembed_add_discovery_links');
                remove_action('wp_head', 'wp_oembed_add_host_js');
                add_filter('embed_oembed_discover', '__return_false');
                add_filter('json_enabled', '__return_false');
                add_filter('json_jsonp_enabled', '__return_false');
                add_filter('rest_enabled', '__return_false');
                add_filter('rest_jsonp_enabled', '__return_false');
                add_filter('rest_authentication_errors', [$this, 'disableRestEnumeration']);
            }
        }

        public function disableRestEnumeration($result)
        {
            if (true === $result || is_wp_error($result)) {
                return $result;
            }

            if (! is_user_logged_in()) {
                return new WP_Error('rest_not_logged_in', __('You are not currently logged in.'), array( 'status' => 401 ));
            }

            return $result;
        }

        private function disableAuthorPages()
        {
            $disable = get_option('tbbi_disable_authors');
            if ($disable) {
                add_action('template_redirect', [$this, 'blockAuthorPages']);
            }
        }

        public function blockAuthorPages()
        {
            global $wp_query;

            if (is_author()) {
                wp_redirect(get_option('home'), 301);
                exit;
            }
        }
    }

    new TrunkBBI();
}
